<?php

class NafathApiService
{
    // Define constants
    private const API_URL = 'https://nafath.api.elm.sa/stg/api/v1/mfa/request';
    private const STATUS_API_URL = 'https://nafath.api.elm.sa/stg/api/v1/mfa/request/status';
    private const JWK_API_URL = 'https://nafath.api.elm.sa/stg/api/v1/mfa/jwk';
    private const APP_ID = '4244d56c';
    private const APP_KEY = '296773f2dbeb3b85e6a287f6c3a35e2d';
    private const SERVICE = 'DigitalServiceEnrollmentWithoutBio';

    private $cookie = 'your_cookie_string_here';

    /**
     * Generate a random unique requestId (16 characters: alphanumeric)
     * 
     * @return string
     */
    private function generateRequestId()
    {
        return bin2hex(random_bytes(8)); // 16-character alphanumeric ID
    }

    /**
     * Send MFA Request
     * 
     * @param string $nationalId The national ID
     * @return array The API response as an array
     */
    public function sendMfaRequest($nationalId)
    {
        $requestId = $this->generateRequestId(); // Generate a unique requestId
        $url = self::API_URL . '?local=ar&requestId=' . urlencode($requestId);

        $payload = json_encode([
            'value' => 'reference #/components/schemas/MfaRequestModel not found in the OpenAPI spec',
            'service' => self::SERVICE,
            'nationalId' => $nationalId,
        ]);

        $response = $this->makeCurlRequest('POST', $url, $payload);

        // Decode the response and return it
        return json_decode($response, true);
    }

    /**
     * Check MFA Request Status
     * 
     * @param string $nationalId The national ID
     * @param string $transId The transaction ID
     * @param string $random A random string
     * @return array The API response as an array
     */
    public function checkMfaRequestStatus($nationalId, $transId, $random)
    {
        $payload = json_encode([
            'nationalId' => $nationalId,
            'transId' => $transId,
            'random' => $random,
        ]);

        $response = $this->makeCurlRequest('POST', self::STATUS_API_URL, $payload);

        // Decode the response and return it
        return json_decode($response, true);
    }

    /**
     * Make a CURL Request
     * 
     * @param string $method HTTP method (GET, POST, etc.)
     * @param string $url API URL
     * @param string|null $payload JSON payload
     * @return string The raw API response
     */
    private function makeCurlRequest($method, $url, $payload = null)
    {
        $curl = curl_init();

        $headers = [
            'APP-ID: ' . self::APP_ID,
            'APP-KEY: ' . self::APP_KEY,
            'Accept: application/json;charset=utf-8',
            'Content-Type: application/json;charset=utf-8',
            'Cookie: ' . $this->cookie,
        ];

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $headers,
        ];

        if ($payload) {
            $options[CURLOPT_POSTFIELDS] = $payload;
        }

        curl_setopt_array($curl, $options);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        curl_close($curl);

        if ($error) {
            return json_encode(['error' => $error]);
        }

        return $response;
    }

    /**
     * Full Workflow: Send MFA Request and Check Status
     * 
     * @param string $nationalId The national ID
     * @return void
     */
    public function sendAndCheckMfaRequest($nationalId) {
        // Send MFA request to get transId and random value
        $mfaResponse = $this->sendMfaRequest($nationalId);

        // Check if the response contains transId and random
        if (isset($mfaResponse['transId']) && isset($mfaResponse['random'])) {
            // Get transId and random values
            $transId = $mfaResponse['transId'];
            $random = $mfaResponse['random'];

            // Show the random value immediately for the user to select it in the app
            echo "Random value for authentication: $random\n";
            echo "Please select the correct random value in your app...\n";

            // Start checking the status
            $this->checkMfaStatusWithDelay($nationalId, $transId, $random);
        } else {
            // Handle error if the response doesn't contain transId or random
            echo "Error: MFA request did not return transId and random.\n";
            echo "Response: " . json_encode($mfaResponse) . "\n";
        }
    }

    // Function to check status with delay
    private function checkMfaStatusWithDelay($nationalId, $transId, $random) {
        // Wait for 2 seconds before checking status
        sleep(2);

        // Call checkMfaRequestStatus function to check status
        $statusResponse = $this->checkMfaRequestStatus($nationalId, $transId, $random);

        // Handle the response from status check
        if (isset($statusResponse['status']) && $statusResponse['status'] == 'COMPLETED') {
            echo "Authentication completed successfully.\n";
        } else if (isset($statusResponse['status']) && $statusResponse['status'] == 'WAITING') {
            echo "Waiting for authentication...\n";
            // Optionally, you can add more logic to recheck the status
        } else {
            echo "Error: " . json_encode($statusResponse) . "\n";
        }
    }

    
    
    
    
}


